import javax.swing.*;
import javax.swing.table.DefaultTableModel;
import java.awt.*;
import java.sql.*;
import java.time.LocalDate;

public class MainApp extends JFrame {
    private String username;
    private JTable table;
    private DefaultTableModel model;

    public MainApp(String username) {
        this.username = username;
        setTitle("CME Tracker - Logged in as " + username);
        setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        setSize(800, 500);
        setLocationRelativeTo(null);

        // Toolbar
        JPanel topPanel = new JPanel(new FlowLayout(FlowLayout.LEFT, 10, 10));
        JButton addButton = new JButton("Add Course");
        JButton reportButton = new JButton("Generate Report");
        JButton refreshButton = new JButton("Refresh");
        JButton aboutButton = new JButton("About");

        topPanel.add(addButton);
        topPanel.add(reportButton);
        topPanel.add(refreshButton);
        topPanel.add(aboutButton);

        // Main table
        model = new DefaultTableModel(new String[]{"Date", "Course Name", "Credit Hours", "Type"}, 0);
        table = new JTable(model);
        add(new JScrollPane(table), BorderLayout.CENTER);
        add(topPanel, BorderLayout.NORTH);

        loadCourses();

        // Button actions
        addButton.addActionListener(e -> addCourse());
        reportButton.addActionListener(e -> showReportDialog());
        refreshButton.addActionListener(e -> loadCourses());
        aboutButton.addActionListener(e -> showAboutDialog());

        setVisible(true);
    }

    private void loadCourses() {
        model.setRowCount(0);
        String sql = "SELECT date, course_name, credit_hours, type FROM Courses WHERE username = ?";
        try (Connection conn = Database.getConnection();
             PreparedStatement ps = conn.prepareStatement(sql)) {
            ps.setString(1, username);
            ResultSet rs = ps.executeQuery();
            while (rs.next()) {
                model.addRow(new Object[]{
                        rs.getString("date"),
                        rs.getString("course_name"),
                        rs.getDouble("credit_hours"),
                        rs.getString("type")
                });
            }
        } catch (SQLException e) {
            e.printStackTrace();
        }
    }

    private void addCourse() {
        JTextField dateField = new JTextField(LocalDate.now().toString());
        JTextField nameField = new JTextField();
        JTextField hoursField = new JTextField();
        JComboBox<String> typeBox = new JComboBox<>(new String[]{"MD", "DO", "Other"});

        JPanel panel = new JPanel(new GridLayout(0, 2));
        panel.add(new JLabel("Date:"));
        panel.add(dateField);
        panel.add(new JLabel("Course Name:"));
        panel.add(nameField);
        panel.add(new JLabel("Credit Hours:"));
        panel.add(hoursField);
        panel.add(new JLabel("Type:"));
        panel.add(typeBox);

        int result = JOptionPane.showConfirmDialog(this, panel, "Add Course",
                JOptionPane.OK_CANCEL_OPTION, JOptionPane.PLAIN_MESSAGE);

        if (result == JOptionPane.OK_OPTION) {
            try (Connection conn = Database.getConnection();
                 PreparedStatement ps = conn.prepareStatement(
                         "INSERT INTO Courses (username, date, course_name, credit_hours, type) VALUES (?, ?, ?, ?, ?)")) {
                ps.setString(1, username);
                ps.setString(2, dateField.getText());
                ps.setString(3, nameField.getText());
                ps.setDouble(4, Double.parseDouble(hoursField.getText()));
                ps.setString(5, (String) typeBox.getSelectedItem());
                ps.executeUpdate();
                loadCourses();
            } catch (SQLException ex) {
                ex.printStackTrace();
                JOptionPane.showMessageDialog(this, "Error saving course: " + ex.getMessage());
            }
        }
    }

    private void showReportDialog() {
        JTextField startField = new JTextField(LocalDate.now().minusMonths(1).toString());
        JTextField endField = new JTextField(LocalDate.now().toString());

        JPanel panel = new JPanel(new GridLayout(0, 2));
        panel.add(new JLabel("Start Date (YYYY-MM-DD):"));
        panel.add(startField);
        panel.add(new JLabel("End Date (YYYY-MM-DD):"));
        panel.add(endField);

        int result = JOptionPane.showConfirmDialog(this, panel, "Select Date Range",
                JOptionPane.OK_CANCEL_OPTION, JOptionPane.PLAIN_MESSAGE);

        if (result == JOptionPane.OK_OPTION) {
            generateDetailedReport(startField.getText(), endField.getText());
        }
    }

    private void generateDetailedReport(String start, String end) {
        DefaultTableModel reportModel = new DefaultTableModel(
                new String[]{"Date", "Course Name", "Credit Hours", "Type"}, 0);

        double totalMD = 0, totalDO = 0, totalOther = 0;

        try (Connection conn = Database.getConnection();
             PreparedStatement ps = conn.prepareStatement(
                     "SELECT date, course_name, credit_hours, type FROM Courses WHERE username=? AND date BETWEEN ? AND ? ORDER BY date ASC")) {

            ps.setString(1, username);
            ps.setString(2, start);
            ps.setString(3, end);
            ResultSet rs = ps.executeQuery();

            while (rs.next()) {
                String type = rs.getString("type");
                double hrs = rs.getDouble("credit_hours");

                switch (type) {
                    case "MD" -> totalMD += hrs;
                    case "DO" -> totalDO += hrs;
                    default -> totalOther += hrs;
                }

                reportModel.addRow(new Object[]{
                        rs.getString("date"),
                        rs.getString("course_name"),
                        hrs,
                        type
                });
            }
        } catch (SQLException e) {
            e.printStackTrace();
            JOptionPane.showMessageDialog(this, "Error generating report: " + e.getMessage());
            return;
        }

        double grandTotal = totalMD + totalDO + totalOther;

        // Summary Text
        StringBuilder summary = new StringBuilder();
        summary.append("Totals by Type:\n");
        summary.append(String.format("  MD    : %.2f hours%n", totalMD));
        summary.append(String.format("  DO    : %.2f hours%n", totalDO));
        summary.append(String.format("  Other : %.2f hours%n", totalOther));
        summary.append(String.format("%nGrand Total: %.2f hours", grandTotal));

        // Build report panel
        JTable reportTable = new JTable(reportModel);
        reportTable.setFillsViewportHeight(true);
        JScrollPane tableScroll = new JScrollPane(reportTable);

        JTextArea summaryArea = new JTextArea(summary.toString());
        summaryArea.setEditable(false);
        summaryArea.setFont(new Font("Monospaced", Font.BOLD, 13));
        summaryArea.setBackground(new Color(245, 245, 245));
        summaryArea.setBorder(BorderFactory.createTitledBorder("Summary"));

        JButton printButton = new JButton("Print Report");
        JButton exportButton = new JButton("Export to Text File");

        printButton.addActionListener(ev -> {
            try {
                boolean printed = reportTable.print();
                if (printed) {
                    JOptionPane.showMessageDialog(this, "Report sent to printer!");
                }
            } catch (Exception ex) {
                JOptionPane.showMessageDialog(this, "Print error: " + ex.getMessage());
            }
        });

        exportButton.addActionListener(ev -> {
            try {
                String fileName = "CME_Report_" + username + "_" + LocalDate.now() + ".txt";
                try (java.io.FileWriter fw = new java.io.FileWriter(fileName)) {
                    fw.write("CME Report for " + username + "\n");
                    fw.write("Date Range: " + start + " to " + end + "\n\n");
                    fw.write(String.format("%-12s | %-30s | %-6s | %-6s%n",
                            "Date", "Course Name", "Hours", "Type"));
                    fw.write("--------------------------------------------------------------\n");
                    for (int i = 0; i < reportModel.getRowCount(); i++) {
                        fw.write(String.format("%-12s | %-30s | %6.2f | %-6s%n",
                                reportModel.getValueAt(i, 0),
                                reportModel.getValueAt(i, 1),
                                reportModel.getValueAt(i, 2),
                                reportModel.getValueAt(i, 3)));
                    }
                    fw.write("\n" + summary);
                }
                JOptionPane.showMessageDialog(this, "Report saved as " + fileName);
            } catch (Exception ex) {
                JOptionPane.showMessageDialog(this, "Export error: " + ex.getMessage());
            }
        });

        JPanel bottomPanel = new JPanel();
        bottomPanel.add(printButton);
        bottomPanel.add(exportButton);

        JPanel mainPanel = new JPanel(new BorderLayout(10, 10));
        mainPanel.add(tableScroll, BorderLayout.CENTER);
        mainPanel.add(summaryArea, BorderLayout.SOUTH);
        mainPanel.add(bottomPanel, BorderLayout.NORTH);

        JDialog dialog = new JDialog(this, "CME Report", true);
        dialog.add(mainPanel);
        dialog.setSize(700, 500);
        dialog.setLocationRelativeTo(this);
        dialog.setVisible(true);
    }

    private void showAboutDialog() {
        ImageIcon logo = new ImageIcon("BBBTransparentLogl.png");
        Image scaledImage = logo.getImage().getScaledInstance(100, 100, Image.SCALE_SMOOTH);
        logo = new ImageIcon(scaledImage);

        JLabel logoLabel = new JLabel(logo, SwingConstants.CENTER);
        JLabel title = new JLabel("CME Tracker", SwingConstants.CENTER);
        title.setFont(new Font("SansSerif", Font.BOLD, 22));
        title.setForeground(new Color(180, 30, 30));
        JLabel author = new JLabel("Developed by Ross Contino DO MSCIS", SwingConstants.CENTER);
        JLabel link = new JLabel("<html><a href='https://bytesbreadbbq.com'>bytesbreadbbq.com</a></html>", SwingConstants.CENTER);
        link.setCursor(Cursor.getPredefinedCursor(Cursor.HAND_CURSOR));
        link.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent e) {
                try {
                    Desktop.getDesktop().browse(new java.net.URI("https://bytesbreadbbq.com"));
                } catch (Exception ignored) {}
            }
        });

        JTextArea info = new JTextArea("""
            Version 1.3
            © 2025 Bytes, Bread, and BBQ
            
            Tracks and reports Continuing Medical Education (CME) hours
            for MDs, DOs, and other professionals.
            """);
        info.setEditable(false);
        info.setBackground(new Color(250, 250, 250));

        JPanel aboutPanel = new JPanel(new BorderLayout(10, 10));
        aboutPanel.add(logoLabel, BorderLayout.NORTH);
        aboutPanel.add(title, BorderLayout.CENTER);
        aboutPanel.add(info, BorderLayout.SOUTH);

        JDialog dialog = new JDialog(this, "About CME Tracker", true);
        dialog.setLayout(new BorderLayout());
        dialog.add(aboutPanel, BorderLayout.CENTER);
        dialog.add(author, BorderLayout.SOUTH);
        dialog.add(link, BorderLayout.NORTH);
        dialog.setSize(400, 450);
        dialog.setLocationRelativeTo(this);
        dialog.setVisible(true);
    }
}
